/**
 * Content script to extract email body from Outlook Web App
 * Called via scripting.executeScript from background.js
 */
(function() {
    // Debug logging - only logs if debugMode is enabled
    const DEBUG = false; // Set via chrome.storage.sync.debugMode in production
    const log = (...args) => DEBUG && console.log('[UltraThink]', ...args);

    log('Starting Outlook content extraction...');

    // Log what we can see on the page for debugging
    const allRoles = document.querySelectorAll('[role]');
    log('Elements with role attribute:', allRoles.length);

    // Try to find email content using multiple strategies
    let emailBody = '';
    let subject = '';
    let from = '';
    let method = 'none';

    // Strategy 1: Look for specific Outlook data-testid attributes
    const messageBody = document.querySelector('[data-testid="messageBody"]');
    if (messageBody) {
        emailBody = messageBody.innerText || '';
        method = 'data-testid messageBody';
        log('Found via data-testid messageBody');
    }

    // Strategy 2: Look for ConversationReadingPaneContainer or similar
    if (!emailBody) {
        const readingPane = document.querySelector('[class*="ConversationReadingPane"], [class*="ReadingPane"], [id*="ReadingPane"]');
        if (readingPane) {
            emailBody = readingPane.innerText || '';
            method = 'ReadingPane class/id';
            log('Found via ReadingPane');
        }
    }

    // Strategy 3: Look for role="document" inside main content
    if (!emailBody) {
        const docs = document.querySelectorAll('[role="document"]');
        for (const doc of docs) {
            const text = doc.innerText || '';
            if (text.length > 100) {
                emailBody = text;
                method = 'role=document';
                log('Found via role=document');
                break;
            }
        }
    }

    // Strategy 4: Look for aria-label containing "message"
    if (!emailBody) {
        const messageRegions = document.querySelectorAll('[aria-label*="message" i], [aria-label*="Message" i]');
        for (const region of messageRegions) {
            const text = region.innerText || '';
            if (text.length > 100) {
                emailBody = text;
                method = 'aria-label message';
                log('Found via aria-label message');
                break;
            }
        }
    }

    // Strategy 5: Look for the largest text block on the page (fallback)
    if (!emailBody || emailBody.length < 100) {
        log('Trying fallback - finding largest text block...');
        const allElements = document.querySelectorAll('div, article, section, main');
        let bestText = '';
        let bestElement = null;

        for (const el of allElements) {
            // Skip navigation, headers, sidebars
            const role = el.getAttribute('role');
            if (role === 'navigation' || role === 'banner' || role === 'complementary') continue;

            const text = el.innerText || '';
            // Look for substantial text that isn't too long (avoid grabbing entire page)
            if (text.length > 200 && text.length < 20000 && text.length > bestText.length) {
                // Check it's not a menu/list (rough heuristic)
                const lineCount = text.split('\n').filter(l => l.trim()).length;
                const avgLineLength = text.length / lineCount;
                if (avgLineLength > 20) { // Email lines tend to be longer than menu items
                    bestText = text;
                    bestElement = el;
                }
            }
        }

        if (bestText && bestText.length > emailBody.length) {
            emailBody = bestText;
            method = 'largest-text-block';
            log('Found via largest text block fallback');
        }
    }

    // Try to extract subject
    const subjectSelectors = [
        '[data-testid="subjectLine"]',
        '[class*="SubjectLine"]',
        '[class*="subjectLine"]',
        'h1[class*="subject" i]',
        '[role="heading"][aria-level="1"]'
    ];
    for (const sel of subjectSelectors) {
        try {
            const el = document.querySelector(sel);
            if (el && el.innerText) {
                subject = el.innerText.trim();
                break;
            }
        } catch (e) {}
    }

    // Try to extract sender
    const fromSelectors = [
        '[data-testid="senderName"]',
        '[class*="SenderName"]',
        '[class*="senderName"]',
        '[class*="FromContainer"]'
    ];
    for (const sel of fromSelectors) {
        try {
            const el = document.querySelector(sel);
            if (el && el.innerText) {
                from = el.innerText.trim();
                break;
            }
        } catch (e) {}
    }

    // Build result
    let result = '';
    if (subject) result += `Subject: ${subject}\n`;
    if (from) result += `From: ${from}\n`;
    if (subject || from) result += '\n';
    result += emailBody.trim();

    // Truncate
    const MAX_LENGTH = 5000;
    if (result.length > MAX_LENGTH) {
        result = result.substring(0, MAX_LENGTH) + '\n\n[Content truncated...]';
    }

    const debug = {
        method: method,
        subject: !!subject,
        from: !!from,
        bodyLength: emailBody.length,
        totalLength: result.length,
        url: window.location.href
    };

    log('Extraction complete:', debug);
    log('First 200 chars:', result.substring(0, 200));

    return {
        emailBody: result,
        subject: subject,
        hasContent: result.trim().length > 50,
        debug: debug
    };
})();
