/**
 * Firebase Configuration for UltraThink Extension
 * Uses Firebase compat SDK for service worker compatibility
 */

// Debug logging - disabled in production
const FIREBASE_DEBUG = false;
const fbConfigLog = (...args) => FIREBASE_DEBUG && console.log('[Firebase]', ...args);

// Firebase configuration (same as React app)
const FIREBASE_CONFIG = {
  apiKey: "AIzaSyAgnrostS9FoSKy0gWhqy5UYFY_wFdnytw",
  authDomain: "ultrathink-50f7f.firebaseapp.com",
  projectId: "ultrathink-50f7f",
  storageBucket: "ultrathink-50f7f.firebasestorage.app",
  messagingSenderId: "463727704482",
  appId: "1:463727704482:web:28f25db94cddc6ee66df45",
  measurementId: "G-Y2WWKY1MBG"
};

// Will be initialized after SDK loads
let firebaseApp = null;
let firebaseDb = null;
let firebaseInitialized = false;

/**
 * Initialize Firebase (called after SDK scripts load)
 */
function initializeFirebase() {
  if (firebaseInitialized) return;

  try {
    // Check if firebase is loaded
    if (typeof firebase === 'undefined') {
      fbConfigLog('SDK not loaded');
      return false;
    }

    // Initialize app
    firebaseApp = firebase.initializeApp(FIREBASE_CONFIG);
    firebaseDb = firebase.firestore();
    firebaseInitialized = true;

    fbConfigLog('Initialized successfully');
    return true;
  } catch (error) {
    fbConfigLog('Initialization error:', error);
    return false;
  }
}

/**
 * Get Firebase credentials from storage
 */
async function getFirebaseCredentials() {
  return new Promise((resolve) => {
    chrome.storage.local.get(['firebaseUserId', 'firebaseDeviceKey'], (result) => {
      resolve({
        userId: result.firebaseUserId || '',
        deviceKey: result.firebaseDeviceKey || ''
      });
    });
  });
}

/**
 * Save Firebase credentials to storage
 */
async function saveFirebaseCredentials(userId, deviceKey) {
  return new Promise((resolve) => {
    chrome.storage.local.set({
      firebaseUserId: userId,
      firebaseDeviceKey: deviceKey
    }, resolve);
  });
}

/**
 * Hash a device key using SHA-256
 */
async function hashDeviceKey(key) {
  const encoder = new TextEncoder();
  const data = encoder.encode(key);
  const hashBuffer = await crypto.subtle.digest('SHA-256', data);
  const hashArray = Array.from(new Uint8Array(hashBuffer));
  return hashArray.map(byte => byte.toString(16).padStart(2, '0')).join('');
}

/**
 * Save entry to Firestore
 * @param {Object} entry - Entry data to save
 * @returns {Promise<Object>} Result with success status
 */
async function saveToFirestore(entry) {
  // Ensure Firebase is initialized
  if (!firebaseInitialized) {
    if (!initializeFirebase()) {
      throw new Error('Firebase not initialized');
    }
  }

  // Get credentials
  const { userId, deviceKey } = await getFirebaseCredentials();

  if (!userId) {
    throw new Error('User ID not configured. Go to extension options to set up Firebase.');
  }

  if (!deviceKey) {
    throw new Error('Device Key not configured. Go to extension options to set up Firebase.');
  }

  // Hash device key for validation
  const deviceKeyHash = await hashDeviceKey(deviceKey);

  // Prepare entry for Firestore
  const firestoreEntry = {
    ...entry,
    userId: userId,
    deviceKeyHash: deviceKeyHash,  // For validation
    visibility: 'private',
    createdAt: firebase.firestore.FieldValue.serverTimestamp(),
    updatedAt: firebase.firestore.FieldValue.serverTimestamp(),
    // Rename 'captured' to 'timestamp' for consistency with existing data
    timestamp: entry.captured,
  };

  // Remove captured (now using timestamp)
  delete firestoreEntry.captured;

  // Add to Firestore
  const docRef = await firebaseDb.collection('entries').add(firestoreEntry);

  fbConfigLog('Entry saved:', docRef.id);

  return {
    success: true,
    id: docRef.id
  };
}

/**
 * Check if Firebase is configured
 */
async function isFirebaseConfigured() {
  const { userId, deviceKey } = await getFirebaseCredentials();
  return !!(userId && deviceKey);
}
