/**
 * Content script to extract conversation from ChatGPT
 * Called via scripting.executeScript from background.js
 */
(function() {
    // Debug logging - only logs if debugMode is enabled in chrome.storage.sync
    const DEBUG = false; // Set via chrome.storage.sync.debugMode in production
    const log = (...args) => DEBUG && console.log('[UltraThink]', ...args);

    log('Starting ChatGPT content extraction...');

    let title = '';
    let method = 'none';

    // Strategy 1: Extract title from active sidebar link
    const activeLink = document.querySelector('a[data-active] span');
    if (activeLink) {
        title = activeLink.innerText.trim();
        log('Found title via active sidebar link');
    }

    // Strategy 2: Fallback - try other title selectors
    if (!title) {
        const titleSelectors = [
            'a[data-active="true"] span',
            '[data-sidebar-item="true"][data-active] span',
            'nav a[href*="/c/"] span'
        ];
        for (const sel of titleSelectors) {
            try {
                const el = document.querySelector(sel);
                if (el && el.innerText) {
                    title = el.innerText.trim();
                    log('Found title via fallback selector:', sel);
                    break;
                }
            } catch (e) {}
        }
    }

    // Extract ALL messages (user and assistant)
    let conversationBody = '';
    const allMessages = [];

    // Strategy 1: Use data-message-author-role attribute (most reliable)
    const messageContainers = document.querySelectorAll('[data-message-author-role]');
    if (messageContainers.length > 0) {
        messageContainers.forEach(container => {
            const role = container.getAttribute('data-message-author-role');
            const textEl = container.querySelector('.whitespace-pre-wrap');
            if (textEl && textEl.innerText && textEl.innerText.trim()) {
                const prefix = role === 'user' ? 'USER:' : 'ASSISTANT:';
                allMessages.push(`${prefix}\n${textEl.innerText.trim()}`);
            }
        });
        if (allMessages.length > 0) {
            conversationBody = allMessages.join('\n\n---\n\n');
            method = 'data-message-author-role';
            log('Extracted', allMessages.length, 'messages via data-message-author-role');
        }
    }

    // Strategy 2: Fallback to user-message-bubble-color (older UI)
    if (!conversationBody) {
        const userMessages = document.querySelectorAll('.user-message-bubble-color .whitespace-pre-wrap');
        if (userMessages.length > 0) {
            userMessages.forEach((el, i) => {
                if (el.innerText && el.innerText.trim()) {
                    allMessages.push(`USER:\n${el.innerText.trim()}`);
                }
            });
            if (allMessages.length > 0) {
                conversationBody = allMessages.join('\n\n---\n\n');
                method = 'user-message-bubble-color';
                log('Extracted', allMessages.length, 'user messages via bubble-color');
            }
        }
    }

    // Truncate if needed (increased limit for full conversations)
    const MAX_LENGTH = 50000;
    if (conversationBody.length > MAX_LENGTH) {
        conversationBody = conversationBody.substring(0, MAX_LENGTH) + '\n\n[Content truncated...]';
    }

    const debug = {
        method: method,
        title: !!title,
        messageCount: allMessages.length,
        conversationLength: conversationBody.length,
        url: window.location.href
    };

    log('ChatGPT extraction complete:', debug);
    log('Title:', title);
    log('First 500 chars:', conversationBody.substring(0, 500));

    return {
        title: title,
        conversationBody: conversationBody,
        hasContent: conversationBody.length > 10,
        debug: debug
    };
})();
