/**
 * @fileoverview UltraThink service worker (background script).
 * Saves entries to Firebase Cloud Function. Screenshot capture and popup coordination.
 * AI processing happens in Cloud Functions (processEntry trigger).
 * @module background
 * @version 3.19.0 - Removed all native messaging (PyQt widget deprecated)
 */

// Import shared constants, logger, and Firebase config
importScripts('shared-constants.js', 'logger.js', 'firebase-config.js');

// Firebase Cloud Function endpoint (2nd Gen - Cloud Run)
const FIREBASE_FUNCTION_URL = 'https://saveentry-kvfxttwv6a-uc.a.run.app';

// Initialize logger and create module-specific loggers
initLogger();
const screenshotLog = createLogger('Screenshot');
const saveLog = createLogger('Save');
const initLog = createLogger('Init');
const metadataLog = createLogger('Metadata');
const firebaseLog = createLogger('Firebase');

/**
 * Get device key from storage
 */
async function getDeviceKey() {
  return new Promise((resolve, reject) => {
    chrome.storage.local.get(['firebaseDeviceKey'], (result) => {
      if (chrome.runtime.lastError) {
        firebaseLog.error('Error reading device key:', chrome.runtime.lastError.message);
        resolve(''); // Return empty string on error to allow graceful degradation
        return;
      }
      resolve(result.firebaseDeviceKey || '');
    });
  });
}

/**
 * Hash a device key using SHA-256
 */
async function hashDeviceKey(key) {
  const encoder = new TextEncoder();
  const data = encoder.encode(key);
  const hashBuffer = await crypto.subtle.digest('SHA-256', data);
  const hashArray = Array.from(new Uint8Array(hashBuffer));
  return hashArray.map(byte => byte.toString(16).padStart(2, '0')).join('');
}

/**
 * Save entry to Firebase via Cloud Function
 * @param {Object} entry - Entry data to save
 * @returns {Promise<Object>} Result with success status
 */
async function saveToFirebase(entry) {
  const deviceKey = await getDeviceKey();

  if (!deviceKey) {
    throw new Error('Device key not configured. Go to extension options to set your device key.');
  }

  // Hash device key
  const deviceKeyHash = await hashDeviceKey(deviceKey);

  // Prepare payload - userId is determined by Cloud Function from device key
  const payload = {
    deviceKeyHash,
    entry: {
      ...entry,
      // Rename 'captured' to 'timestamp' for consistency
      timestamp: entry.captured,
      visibility: 'private',
    }
  };
  delete payload.entry.captured;

  firebaseLog.info('Saving to Firebase...', { type: entry.type, title: entry.title?.substring(0, 50) });

  const response = await fetch(FIREBASE_FUNCTION_URL, {
    method: 'POST',
    headers: {
      'Content-Type': 'application/json',
    },
    body: JSON.stringify(payload),
  });

  const result = await response.json();

  if (!response.ok) {
    throw new Error(result.error || 'Failed to save to Firebase');
  }

  firebaseLog.info('Entry saved to Firebase:', result.id);
  return result;
}

/**
 * Check if Firebase is configured
 */
async function isFirebaseConfigured() {
  const deviceKey = await getDeviceKey();
  return !!deviceKey;
}

/**
 * Detects the content type based on URL patterns.
 * Identifies AI tools (Claude, ChatGPT, Perplexity), documents, and media.
 *
 * @function detectUrlType
 * @param {string} url - The URL to analyse
 * @returns {string} Content type identifier (e.g., 'claude', 'pdf', 'video', 'link')
 */
function detectUrlType(url) {
  if (!url) return 'link';
  const urlLower = url.toLowerCase();

  // AI conversation types
  if (urlLower.includes('claude.ai')) return 'claude';
  if (urlLower.includes('chat.openai.com') || urlLower.includes('chatgpt.com')) return 'chatgpt';
  if (urlLower.includes('perplexity.ai')) return 'perplexity';

  // Outlook emails
  if (urlLower.includes('outlook.office.com/mail') || urlLower.includes('outlook.office365.com/mail') || urlLower.includes('outlook.live.com/mail')) return 'outlook';

  // PDF files
  if (urlLower.endsWith('.pdf') || urlLower.includes('.pdf?') || urlLower.includes('/pdf/')) return 'pdf';

  // Markdown files
  if (urlLower.endsWith('.md') || urlLower.includes('.md?')) return 'markdown';

  // Microsoft Office - SharePoint patterns
  if (urlLower.includes('sharepoint.com') || urlLower.includes('1drv.ms') || urlLower.includes('onedrive.live.com')) {
    if (urlLower.includes(':w:') || urlLower.includes('/_layouts/15/doc.aspx')) return 'ms-word';
    if (urlLower.includes(':p:')) return 'ms-powerpoint';
    if (urlLower.includes(':x:')) return 'ms-excel';
    if (urlLower.includes(':b:') || urlLower.includes('onenote.aspx')) return 'ms-onenote';
    return 'ms-word';  // Default to Word if unclear
  }

  // Notion
  if (urlLower.includes('notion.so') || urlLower.includes('notion.site')) return 'notion';

  // Video
  if (urlLower.includes('youtube.com') || urlLower.includes('youtu.be') || urlLower.includes('vimeo.com')) return 'video';

  // Default
  return 'link';
}

/**
 * Temporary storage for screenshot data between capture and popup.
 * Set when screenshot is taken, cleared when popup retrieves it.
 * @type {Object|null}
 */
let pendingScreenshot = null;

/**
 * Guard to prevent multiple concurrent screenshot operations.
 * Reset after operation completes or times out.
 */
let screenshotInProgress = false;

// Handle keyboard commands
chrome.commands.onCommand.addListener((command) => {
  screenshotLog.info('>>> Command received:', command, '| guard:', screenshotInProgress);
  if (command === 'capture-screenshot') {
    // Simple debounce - just prevent rapid double-triggers (500ms)
    if (screenshotInProgress) {
      screenshotLog.info('Screenshot debounce - ignoring rapid trigger');
      return;
    }
    screenshotInProgress = true;
    screenshotLog.info('Starting screenshot selection...');
    startScreenshotSelection();

    // Quick reset after 500ms - just prevent double-triggers, not long locks
    setTimeout(() => {
      screenshotInProgress = false;
    }, 500);
  }
});

// Handle messages from popup
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
  if (request.action === 'save') {
    handleSaveSingle(request)
      .then(result => sendResponse(result))
      .catch(error => sendResponse({ success: false, error: error.message }));
    return true; // Keep channel open for async response
  } else if (request.action === 'saveAllTabs') {
    handleSaveAllTabs(request)
      .then(result => sendResponse(result))
      .catch(error => sendResponse({ success: false, error: error.message }));
    return true;
  } else if (request.action === 'getScreenshot') {
    sendResponse({ screenshot: pendingScreenshot });
    pendingScreenshot = null; // Clear after retrieval
    return true;
  } else if (request.action === 'screenshotCancelled') {
    screenshotLog.info('Screenshot cancelled by user (ESC)');
    screenshotInProgress = false;
    sendResponse({ success: true });
    return true;
  } else if (request.action === 'areaSelected') {
    screenshotLog.info('>>> Message: areaSelected, rect:', JSON.stringify(request.rect), 'from tab:', sender.tab?.id);
    captureAreaScreenshot(request.rect, sender.tab)
      .then(() => {
        screenshotLog.info('<<< areaSelected completed successfully');
        sendResponse({ success: true });
      })
      .catch(error => {
        screenshotLog.error('<<< areaSelected failed:', error.message);
        sendResponse({ success: false, error: error.message });
      });
    return true;
  } else if (request.action === 'captureFullScreen') {
    screenshotLog.info('>>> Message: captureFullScreen from tab:', sender.tab?.id);
    captureScreenshot(sender.tab)
      .then(() => {
        screenshotLog.info('<<< captureFullScreen completed successfully');
        sendResponse({ success: true });
      })
      .catch(error => {
        screenshotLog.error('<<< captureFullScreen failed:', error.message);
        sendResponse({ success: false, error: error.message });
      });
    return true;
  } else if (request.action === 'saveFile') {
    handleFileSave(request, sender.tab)
      .then(result => sendResponse(result))
      .catch(error => sendResponse({ success: false, error: error.message }));
    return true;
  }
});

/**
 * Transforms an Outlook URL to a proper deep link that won't redirect to inbox.
 * Extracts the message ID and constructs a permanent deep link.
 *
 * @function transformOutlookUrl
 * @param {string} url - The original Outlook URL
 * @returns {string} The transformed deep link URL, or original if no ID found
 * @example
 * transformOutlookUrl('https://outlook.office.com/mail/inbox/id/AAkALg...')
 * // Returns: 'https://outlook.office365.com/owa/?ItemID=AAkALg...&exvsurl=1&viewmodel=ReadMessageItem'
 */
function transformOutlookUrl(url) {
  if (!url) return url;

  // Extract message ID from URL patterns like /mail/inbox/id/MESSAGE_ID or /mail/id/MESSAGE_ID
  const idMatch = url.match(/\/mail\/(?:[^/]+\/)?id\/([^/?#]+)/);
  if (!idMatch) {
    saveLog.debug('No message ID found in Outlook URL, keeping original');
    return url;
  }

  const messageId = idMatch[1];

  // Determine the base URL based on the original URL
  let baseUrl;
  if (url.includes('outlook.live.com')) {
    baseUrl = 'https://outlook.live.com/owa/';
  } else {
    // office.com and office365.com both use office365.com for deep links
    baseUrl = 'https://outlook.office365.com/owa/';
  }

  const deepLink = `${baseUrl}?ItemID=${encodeURIComponent(messageId)}&exvsurl=1&viewmodel=ReadMessageItem`;
  saveLog.info(`Transformed Outlook URL to deep link: ${deepLink.substring(0, 80)}...`);
  return deepLink;
}

/**
 * Retrieves tab group information for a given tab.
 * Returns group name and colour if the tab belongs to a group.
 *
 * @async
 * @function getTabGroupInfo
 * @param {chrome.tabs.Tab} tab - The tab to get group info for
 * @returns {Promise<Object|null>} Object with groupName and groupColor, or null
 */
async function getTabGroupInfo(tab) {
  if (tab.groupId && tab.groupId !== -1) {
    try {
      const group = await chrome.tabGroups.get(tab.groupId);
      return {
        groupName: group.title || '',
        groupColor: group.color
      };
    } catch (error) {
      saveLog.error('Error getting tab group:', error);
      return null;
    }
  }
  return null;
}

/**
 * Extracts page metadata by injecting page-metadata.js into the tab.
 * Retrieves description, og:image, author, published date, and reading time.
 *
 * @async
 * @function getPageMetadata
 * @param {number} tabId - The ID of the tab to extract metadata from
 * @returns {Promise<Object|null>} Metadata object or null on failure
 */
async function getPageMetadata(tabId) {
  try {
    metadataLog.debug('Extracting metadata for tab:', tabId);

    const results = await chrome.scripting.executeScript({
      target: { tabId: tabId },
      files: ['page-metadata.js']
    });

    if (results && results[0] && results[0].result) {
      metadataLog.debug('Metadata extracted:', results[0].result);
      return results[0].result;
    }

    return null;
  } catch (error) {
    metadataLog.error('Error extracting metadata:', error);
    return null;
  }
}

/**
 * Handles saving a single tab entry to the knowledge base.
 * Extracts metadata, builds entry object, and sends to native host.
 *
 * @async
 * @function handleSaveSingle
 * @param {Object} request - Save request with tab, type, notes, selectedText
 * @returns {Promise<Object>} Result object with success status
 */
async function handleSaveSingle(request) {
  try {
    const timestamp = formatTimestamp();
    const tab = request.tab;
    const tabGroup = await getTabGroupInfo(tab);

    // Transform Outlook URLs to deep links
    let finalUrl = tab.url;
    if (request.type === 'outlook') {
      finalUrl = transformOutlookUrl(tab.url);
    }

    // Build entry with new consistent format
    const data = {
      type: request.type,
      source: 'browser',           // New: 'browser' or 'widget'
      captured: timestamp,
      title: tab.title,
      url: finalUrl,               // Transformed for Outlook, original otherwise
      tabGroup: tabGroup,
      selectedText: request.selectedText || '',  // Separate: text selected from page
      notes: request.notes || ''                 // User notes/commentary (canonical field)
    };

    // Log entry details
    saveLog.info(`[${timestamp}] Saving entry:`, {
      type: request.type,
      title: tab.title.substring(0, 50) + (tab.title.length > 50 ? '...' : ''),
      hasSelectedText: !!request.selectedText,
      hasNotes: !!request.notes,
      notes: request.notes ? `"${request.notes.substring(0, 50)}..."` : '(empty)',
      tabGroup: tabGroup?.groupName || null
    });

    // Extract page metadata for link-type entries (not screenshots, not special pages)
    const linkTypes = ['link', 'claude', 'chatgpt', 'perplexity', 'pdf', 'markdown', 'notion', 'video', 'ms-word', 'ms-excel', 'ms-powerpoint', 'ms-onenote', 'outlook'];
    if (linkTypes.includes(request.type) && tab.url && tab.url.startsWith('http')) {
      try {
        const metadata = await getPageMetadata(tab.id);
        if (metadata) {
          data.pageMetadata = metadata;
          metadataLog.info(`[${timestamp}] Page metadata extracted:`, {
            hasDescription: !!metadata.description,
            hasOgImage: !!metadata.ogImage,
            author: metadata.author || null,
            readingTime: metadata.readingTime || null
          });
        }
      } catch (e) {
        metadataLog.debug('Could not extract metadata:', e.message);
      }
    }

    // Extract email body for Outlook entries
    if (request.type === 'outlook' && tab.url && tab.url.startsWith('http')) {
      try {
        saveLog.info(`[${timestamp}] Attempting Outlook email body extraction...`);
        const results = await chrome.scripting.executeScript({
          target: { tabId: tab.id },
          files: ['outlook-content.js']
        });
        saveLog.info(`[${timestamp}] Outlook script result:`, results?.[0]?.result?.debug || 'no debug info');
        if (results && results[0] && results[0].result && results[0].result.hasContent) {
          data.emailBody = results[0].result.emailBody;
          // Use subject as title if available
          if (results[0].result.subject) {
            data.title = results[0].result.subject;
          }
          saveLog.info(`[${timestamp}] Outlook email body extracted (${data.emailBody.length} chars)`);
        } else {
          saveLog.info(`[${timestamp}] Outlook email body extraction returned no content`);
        }
      } catch (e) {
        saveLog.error(`[${timestamp}] Could not extract Outlook email body:`, e.message);
      }
    }

    // Notion - auto-capture screenshot for AI content extraction
    if (request.type === 'notion' && tab.url && tab.url.startsWith('http')) {
      try {
        saveLog.info(`[${timestamp}] Auto-capturing Notion page screenshot...`);

        // Capture visible viewport
        const windowId = tab.windowId;
        const dataUrl = await chrome.tabs.captureVisibleTab(windowId, { format: 'png' });

        // Compress to JPEG (reuse existing compressImage function)
        const compressedDataUrl = await compressImage(dataUrl);

        // Attach to entry - backend will process with AI vision
        data.screenshot = compressedDataUrl;
        data.extractContent = true; // Flag for backend to use content extraction prompt

        saveLog.info(`[${timestamp}] Notion screenshot captured (${Math.round(compressedDataUrl.length / 1024)}KB)`);
      } catch (e) {
        saveLog.error(`[${timestamp}] Could not capture Notion screenshot:`, e.message);
        // Continue saving without screenshot - graceful degradation
      }
    }

    // Extract conversation for ChatGPT entries
    if (request.type === 'chatgpt' && tab.url && tab.url.startsWith('http')) {
      try {
        saveLog.info(`[${timestamp}] Attempting ChatGPT conversation extraction...`);
        const results = await chrome.scripting.executeScript({
          target: { tabId: tab.id },
          files: ['chatgpt-content.js']
        });
        saveLog.info(`[${timestamp}] ChatGPT script result:`, results?.[0]?.result?.debug || 'no debug info');
        if (results && results[0] && results[0].result && results[0].result.hasContent) {
          data.conversationBody = results[0].result.conversationBody;
          // Use extracted title if available, unless page title is more descriptive
          if (results[0].result.title) {
            const pageTitle = data.title || '';
            const extractedTitle = results[0].result.title;
            
            // Check if page title is generic or less descriptive
            const genericTitles = ['chatgpt', 'new chat', 'untitled', 'chat'];
            const isGenericTitle = genericTitles.some(generic => 
              pageTitle.toLowerCase().includes(generic)
            );
            
            // Use extracted title if:
            // 1. No page title exists
            // 2. Page title is generic
            // 3. Page title is very short (likely incomplete)
            // 4. Page title ends with ".." (truncated)
            if (!pageTitle || 
                isGenericTitle || 
                pageTitle.length < 10 ||
                pageTitle.endsWith('..') ||
                pageTitle.endsWith('…')) {
              saveLog.info(`[${timestamp}] Using extracted sidebar title: "${extractedTitle}" instead of page title: "${pageTitle}"`);
              data.title = extractedTitle;
            } else {
              saveLog.info(`[${timestamp}] Keeping page title: "${pageTitle}" (extracted: "${extractedTitle}")`);
            }
          }
          saveLog.info(`[${timestamp}] ChatGPT conversation extracted (${data.conversationBody.length} chars)`);
        } else {
          saveLog.info(`[${timestamp}] ChatGPT conversation extraction returned no content`);
        }
      } catch (e) {
        saveLog.error(`[${timestamp}] Could not extract ChatGPT conversation:`, e.message);
      }
    }

    // Add screenshot data if present
    if (request.screenshotData && request.type === 'screenshot') {
      data.screenshot = request.screenshotData.dataUrl;
      saveLog.info(`[${timestamp}] Screenshot attached (${Math.round(request.screenshotData.dataUrl.length / 1024)}KB)`);
    }

    // Save to Firebase (cloud-only mode - v3.18.0)
    saveLog.info(`[${timestamp}] Saving to Firebase...`);
    const response = await saveToFirebase(data);

    if (response && response.success) {
      saveLog.info(`[${timestamp}] Entry saved to Firebase (ID: ${response.id})`);
      return { success: true };
    } else {
      throw new Error(response?.error || 'Firebase save failed');
    }
  } catch (error) {
    saveLog.error('Save error:', error);
    throw error;
  }
}

/**
 * Handles bulk saving of all tabs in the current window.
 * Saves to Firebase. Auto-links entries as related.
 *
 * @async
 * @function handleSaveAllTabs
 * @param {Object} request - Request with tabs array
 * @returns {Promise<Object>} Result object with success status and count
 */
async function handleSaveAllTabs(request) {
  try {
    const timestamp = formatTimestamp();
    const tabs = request.tabs;
    let savedCount = 0;

    saveLog.info(`[${timestamp}] Bulk save starting: ${tabs.length} tabs`);

    // Step 1: Pre-generate timestamps and build entry data for all tabs
    const entries = tabs.map(tab => {
      const tabTimestamp = formatTimestamp();
      const detectedType = detectUrlType(tab.url);

      return {
        timestamp: tabTimestamp,
        tab: tab,
        data: {
          type: detectedType,
          source: 'browser',
          captured: tabTimestamp,
          title: tab.title,
          url: tab.url,
          tabGroup: null,  // Will be populated below
          selectedText: '',
          content: '',
          related: []  // Will be populated below
        }
      };
    });

    // Step 2: Populate `related` arrays - each entry links to all OTHER entries
    for (let i = 0; i < entries.length; i++) {
      entries[i].data.related = entries
        .filter((_, j) => j !== i)  // Exclude self
        .map(e => ({
          timestamp: e.timestamp,
          type: 'saved-together'
        }));
    }

    // Step 3: Save all entries to Firebase (cloud-only mode - v3.18.0)
    for (const entry of entries) {
      try {
        // Get tab group info
        entry.data.tabGroup = await getTabGroupInfo(entry.tab);

        // Save to Firebase
        await saveToFirebase(entry.data);
        savedCount++;
        saveLog.info(`[${entry.timestamp}] Saved tab ${savedCount}/${tabs.length} to Firebase: ${entry.tab.title.substring(0, 40)}...`);
      } catch (e) {
        saveLog.error(`Failed to save tab: ${entry.tab.title}`, e.message);
      }
    }

    saveLog.info(`[${timestamp}] Bulk save complete: ${savedCount}/${tabs.length} tabs saved`);
    if (entries.length > 1) {
      saveLog.info(`[${timestamp}] Entries linked via 'related' field (type: saved-together)`);
    }

    return { success: true, count: savedCount };
  } catch (error) {
    saveLog.error('Bulk save error:', error);
    throw error;
  }
}

/**
 * Handles saving a dropped/uploaded file to the knowledge base.
 * Supports files and pasted text from the pinned dialog.
 *
 * @async
 * @function handleFileSave
 * @param {Object} request - Request with file data, type, and metadata
 * @param {chrome.tabs.Tab} tab - The current tab for context
 * @returns {Promise<Object>} Result object with success status
 */
async function handleFileSave(request, tab) {
  try {
    const timestamp = formatTimestamp();
    const originalNotes = request.notes || '';

    // Build entry with new consistent format
    let data = {
      type: request.detectedType || 'file',  // Use detected type (pdf, image, audio, etc.)
      source: 'browser',                      // Pinned dialog is in browser context
      captured: timestamp,
      title: request.fileName,
      url: '',                                // No URL for dropped files
      tabGroup: null,                         // No tab group for dropped files
      selectedText: '',
      notes: originalNotes                    // User notes/commentary (canonical field)
    };

    // Add file data if it's a file (not text)
    if (request.fileType === 'file') {
      data.fileData = request.fileData;
      data.mimeType = request.mimeType;
      const fileSizeKB = Math.round(request.fileData.length / 1024);
      saveLog.info(`[${timestamp}] Saving file:`, {
        type: data.type,
        fileName: request.fileName,
        mimeType: request.mimeType,
        sizeKB: fileSizeKB,
        hasNotes: !!originalNotes
      });
    } else if (request.fileType === 'text') {
      // For pasted text, save as snippet with page context
      data.type = 'snippet';
      data.url = tab.url;              // Use current page URL for text
      data.title = tab.title;
      data.selectedText = request.content;  // Pasted text goes to selectedText
      saveLog.info(`[${timestamp}] Saving pasted text:`, {
        type: 'snippet',
        contentLength: request.content.length,
        pageTitle: tab.title.substring(0, 40) + '...',
        hasNotes: !!originalNotes
      });
    }

    // Save to Firebase (cloud-only mode - v3.18.0)
    saveLog.info(`[${timestamp}] Saving file to Firebase...`);
    const response = await saveToFirebase(data);

    if (response && response.success) {
      saveLog.info(`[${timestamp}] File saved to Firebase (ID: ${response.id})`);
      return { success: true };
    } else {
      throw new Error(response?.error || 'Firebase save failed');
    }
  } catch (error) {
    saveLog.error('File save error:', error);
    throw error;
  }
}

/**
 * Initiates the screenshot selection process.
 * Injects the selection overlay script into the active tab.
 *
 * @async
 * @function startScreenshotSelection
 */
async function startScreenshotSelection() {
  try {
    screenshotLog.info('>>> startScreenshotSelection called');

    // Try to get active tab in current window
    let [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
    screenshotLog.info('Active tab (currentWindow):', tab?.id, tab?.url?.substring(0, 50));

    // Fallback: try lastFocusedWindow if no tab found
    if (!tab) {
      screenshotLog.info('No tab in currentWindow, trying lastFocusedWindow...');
      [tab] = await chrome.tabs.query({ active: true, lastFocusedWindow: true });
      screenshotLog.info('Active tab (lastFocusedWindow):', tab?.id, tab?.url?.substring(0, 50));
    }

    // Still no tab? Log error and return
    if (!tab) {
      screenshotLog.error('No active tab found in any window');
      return;
    }

    // Inject selection overlay
    screenshotLog.info('Injecting selection-overlay.js into tab:', tab.id);
    await chrome.scripting.executeScript({
      target: { tabId: tab.id },
      files: ['selection-overlay.js']
    });
    screenshotLog.info('<<< Selection overlay injected successfully');
  } catch (error) {
    screenshotLog.error('Selection overlay injection error:', error.message, error.stack);
    screenshotInProgress = false;
  }
}

/**
 * Captures a full-page screenshot of the visible tab.
 * Stores result in pendingScreenshot and opens the popup.
 *
 * @async
 * @function captureScreenshot
 * @param {chrome.tabs.Tab} [tab] - Tab to capture, defaults to active tab
 */
async function captureScreenshot(tab) {
  try {
    screenshotLog.info('>>> captureScreenshot called, tab:', tab?.id, tab?.url?.substring(0, 50));
    if (!tab) {
      screenshotLog.info('No tab provided, querying active tab...');
      [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
      screenshotLog.info('Got active tab:', tab?.id, tab?.url?.substring(0, 50));
    }

    // Capture visible tab
    screenshotLog.info('Calling captureVisibleTab for windowId:', tab.windowId);
    const dataUrl = await chrome.tabs.captureVisibleTab(tab.windowId, { format: 'png' });
    screenshotLog.info('Screenshot captured, dataUrl length:', dataUrl.length);

    // Compress image to reduce size for Firestore (1MB limit)
    screenshotLog.info('Compressing image...');
    const compressedDataUrl = await compressImage(dataUrl);
    screenshotLog.info('Compressed to:', compressedDataUrl.length);

    // Store screenshot data
    pendingScreenshot = {
      dataUrl: compressedDataUrl,
      tabId: tab.id,
      url: tab.url,
      title: tab.title
    };
    screenshotLog.info('Stored pendingScreenshot, dataUrl length:', pendingScreenshot.dataUrl.length);

    // Open popup
    screenshotLog.info('>>> Calling chrome.action.openPopup()...');
    const result = await chrome.action.openPopup();
    screenshotLog.info('<<< openPopup returned:', result);
    screenshotInProgress = false;
  } catch (error) {
    screenshotLog.error('Screenshot capture error:', error.message, error.stack);
    screenshotInProgress = false;
  }
}

/**
 * Captures a screenshot of a user-selected area.
 * Takes full screenshot then crops to the specified rectangle.
 *
 * @async
 * @function captureAreaScreenshot
 * @param {Object} rect - Selection rectangle {x, y, width, height}
 * @param {chrome.tabs.Tab} tab - Tab where selection was made
 */
async function captureAreaScreenshot(rect, tab) {
  try {
    screenshotLog.info('>>> captureAreaScreenshot called, rect:', JSON.stringify(rect), 'tab:', tab?.id);

    // Capture full visible tab first
    screenshotLog.info('Calling captureVisibleTab for windowId:', tab.windowId);
    const dataUrl = await chrome.tabs.captureVisibleTab(tab.windowId, { format: 'png' });
    screenshotLog.info('Full screenshot captured, dataUrl length:', dataUrl.length);

    // Crop to selected area using canvas
    screenshotLog.info('Cropping image...');
    const croppedDataUrl = await cropImage(dataUrl, rect);
    screenshotLog.info('Cropped to:', croppedDataUrl.length);

    // Store screenshot data
    pendingScreenshot = {
      dataUrl: croppedDataUrl,
      tabId: tab.id,
      url: tab.url,
      title: tab.title
    };
    screenshotLog.info('Stored pendingScreenshot, dataUrl length:', pendingScreenshot.dataUrl.length);

    // Open popup
    screenshotLog.info('>>> Calling chrome.action.openPopup()...');
    const result = await chrome.action.openPopup();
    screenshotLog.info('<<< openPopup returned:', result);
    screenshotInProgress = false;
  } catch (error) {
    screenshotLog.error('Area screenshot capture error:', error.message, error.stack);
    screenshotInProgress = false;
  }
}

/**
 * Compresses an image data URL to JPEG with quality reduction.
 * Reduces size to fit within Firestore's 1MB document limit.
 *
 * @async
 * @function compressImage
 * @param {string} dataUrl - Base64 data URL of the image
 * @param {number} maxWidth - Maximum width (default 1920)
 * @param {number} quality - JPEG quality 0-1 (default 0.7)
 * @returns {Promise<string>} Compressed image as data URL
 */
async function compressImage(dataUrl, maxWidth = 1920, quality = 0.7) {
  try {
    // Convert data URL to blob
    const response = await fetch(dataUrl);
    const blob = await response.blob();

    // Decode image
    const imageBitmap = await createImageBitmap(blob);
    let { width, height } = imageBitmap;

    // Scale down if needed
    if (width > maxWidth) {
      const scale = maxWidth / width;
      width = maxWidth;
      height = Math.round(height * scale);
    }

    // Create canvas
    const canvas = new OffscreenCanvas(width, height);
    const ctx = canvas.getContext('2d');
    ctx.drawImage(imageBitmap, 0, 0, width, height);

    // Convert to JPEG with compression
    const compressedBlob = await canvas.convertToBlob({ type: 'image/jpeg', quality });

    // Convert to data URL
    const reader = new FileReader();
    const compressedDataUrl = await new Promise((resolve, reject) => {
      reader.onloadend = () => resolve(reader.result);
      reader.onerror = reject;
      reader.readAsDataURL(compressedBlob);
    });

    screenshotLog.debug(`Compressed: ${Math.round(dataUrl.length/1024)}KB -> ${Math.round(compressedDataUrl.length/1024)}KB`);
    return compressedDataUrl;
  } catch (error) {
    screenshotLog.error('Compression error:', error);
    return dataUrl; // Fallback to original
  }
}

/**
 * Crops an image data URL to a specified rectangle.
 * Uses OffscreenCanvas for service worker compatibility.
 *
 * @async
 * @function cropImage
 * @param {string} dataUrl - Base64 data URL of the full screenshot
 * @param {Object} rect - Crop rectangle {x, y, width, height}
 * @returns {Promise<string>} Cropped image as data URL
 */
async function cropImage(dataUrl, rect) {
  try {
    screenshotLog.debug('cropImage called with rect:', rect);

    // Convert data URL to blob
    const response = await fetch(dataUrl);
    const blob = await response.blob();

    // Decode image using createImageBitmap (service worker API)
    const imageBitmap = await createImageBitmap(blob);
    screenshotLog.debug('Image loaded, dimensions:', imageBitmap.width, 'x', imageBitmap.height);

    // Create OffscreenCanvas (available in service workers)
    const canvas = new OffscreenCanvas(rect.width, rect.height);
    const ctx = canvas.getContext('2d');

    screenshotLog.debug('Canvas created:', canvas.width, 'x', canvas.height);

    // Draw cropped portion
    ctx.drawImage(
      imageBitmap,
      rect.x, rect.y, rect.width, rect.height,  // source rectangle
      0, 0, rect.width, rect.height              // destination rectangle
    );

    screenshotLog.debug('Image drawn to canvas');

    // Convert to JPEG with compression (not PNG) to reduce size
    const croppedBlob = await canvas.convertToBlob({ type: 'image/jpeg', quality: 0.8 });

    // Convert blob to data URL
    const reader = new FileReader();
    const croppedDataUrl = await new Promise((resolve, reject) => {
      reader.onloadend = () => resolve(reader.result);
      reader.onerror = reject;
      reader.readAsDataURL(croppedBlob);
    });

    screenshotLog.debug('Cropped image dataUrl length:', croppedDataUrl.length);
    return croppedDataUrl;

  } catch (error) {
    screenshotLog.error('Image crop error:', error);
    throw error;
  }
}

// Initialize default settings on install
chrome.runtime.onInstalled.addListener(() => {
  chrome.storage.sync.set(DEFAULT_SETTINGS);
  initLog.info('UltraThink extension installed');
});

